<?php
// ARCHIVO: #2 - app/Models/Condominio.php
// VERSIÓN: 1.0
// PREDECESORES: User.php (#1)
// DESCRIPCIÓN: Modelo para gestión completa de condominios/edificios
// RUTA EXACTA: app/Models/Condominio.php
// COMPATIBILIDAD: Laravel 11.47, PHP 8.2+, MySQL 5.5+
// FECHA: 2024-01-01
// AUTOR: UrbanOSS Platform

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Storage;

class Condominio extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'condominios';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'nombre',
        'direccion',
        'logo',
        'moneda',
        'idioma',
        'telefono',
        'email',
        'estado',
        'superadministrador_id',
        'configuracion',
        'observaciones'
    ];

    /**
     * The attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'configuracion' => 'array',
            'created_at' => 'datetime',
            'updated_at' => 'datetime',
            'deleted_at' => 'datetime',
        ];
    }

    /**
     * Relación con el superadministrador que creó el condominio.
     */
    public function superadministrador(): BelongsTo
    {
        return $this->belongsTo(User::class, 'superadministrador_id');
    }

    /**
     * Relación con usuarios del condominio.
     */
    public function usuarios(): HasMany
    {
        return $this->hasMany(User::class);
    }

    /**
     * Relación con unidades del condominio.
     */
    public function unidades(): HasMany
    {
        return $this->hasMany(Unidad::class);
    }

    /**
     * Relación con áreas comunes del condominio.
     */
    public function areasComunes(): HasMany
    {
        return $this->hasMany(AreaComun::class);
    }

    /**
     * Relación con cámaras de seguridad del condominio.
     */
    public function camaras(): HasMany
    {
        return $this->hasMany(Camara::class);
    }

    /**
     * Relación con accesos registrados en el condominio.
     */
    public function accesos(): HasMany
    {
        return $this->hasMany(Acceso::class);
    }

    /**
     * Relación con reclamaciones del condominio.
     */
    public function reclamaciones(): HasMany
    {
        return $this->hasMany(Reclamacion::class);
    }

    /**
     * Relación con pagos del condominio (a través de unidades).
     */
    public function pagos()
    {
        return $this->hasManyThrough(Pago::class, Unidad::class);
    }

    /**
     * Relación con reservas (a través de áreas comunes).
     */
    public function reservas()
    {
        return $this->hasManyThrough(Reserva::class, AreaComun::class);
    }

    /**
     * Obtiene los administradores del condominio.
     */
    public function administradores()
    {
        return $this->usuarios()->where('role', 'admin');
    }

    /**
     * Obtiene los propietarios del condominio.
     */
    public function propietarios()
    {
        return $this->usuarios()->where('role', 'propietario');
    }

    /**
     * Obtiene los inquilinos del condominio.
     */
    public function inquilinos()
    {
        return $this->usuarios()->where('role', 'inquilino');
    }

    /**
     * Obtiene los guardias del condominio.
     */
    public function guardias()
    {
        return $this->usuarios()->where('role', 'guardia');
    }

    /**
     * Verifica si el condominio está activo.
     */
    public function isActive(): bool
    {
        return $this->estado === 'activo';
    }

    /**
     * Verifica si el condominio está inactivo.
     */
    public function isInactive(): bool
    {
        return $this->estado === 'inactivo';
    }

    /**
     * Verifica si el condominio está suspendido.
     */
    public function isSuspended(): bool
    {
        return $this->estado === 'suspendido';
    }

    /**
     * Obtiene el estado formateado.
     */
    public function getEstadoDisplayAttribute(): string
    {
        $estados = [
            'activo' => 'Activo',
            'inactivo' => 'Inactivo',
            'suspendido' => 'Suspendido',
        ];

        return $estados[$this->estado] ?? ucfirst($this->estado);
    }

    /**
     * Obtiene el idioma formateado.
     */
    public function getIdiomaDisplayAttribute(): string
    {
        $idiomas = [
            'es' => 'Español',
            'en' => 'Inglés',
        ];

        return $idiomas[$this->idioma] ?? strtoupper($this->idioma);
    }

    /**
     * Obtiene la moneda formateada.
     */
    public function getMonedaDisplayAttribute(): string
    {
        $monedas = [
            'PEN' => 'Soles (PEN)',
            'USD' => 'Dólares (USD)',
        ];

        return $monedas[$this->moneda] ?? $this->moneda;
    }

    /**
     * Obtiene el símbolo de moneda.
     */
    public function getMonedaSimboloAttribute(): string
    {
        return $this->moneda === 'USD' ? '$' : 'S/';
    }

    /**
     * Obtiene la URL del logo.
     */
    public function getLogoUrlAttribute(): ?string
    {
        if (!$this->logo) {
            return null;
        }

        if (filter_var($this->logo, FILTER_VALIDATE_URL)) {
            return $this->logo;
        }

        if (Storage::disk('public')->exists($this->logo)) {
            return Storage::disk('public')->url($this->logo);
        }

        return asset('storage/' . $this->logo);
    }

    /**
     * Obtiene estadísticas del condominio.
     */
    public function getEstadisticasAttribute(): array
    {
        return [
            'unidades_totales' => $this->unidades()->count(),
            'unidades_ocupadas' => $this->unidades()->where('estado', 'ocupado')->count(),
            'unidades_vacias' => $this->unidades()->where('estado', 'vacio')->count(),
            'unidades_airbnb' => $this->unidades()->where('estado', 'airbnb')->count(),
            'usuarios_totales' => $this->usuarios()->count(),
            'propietarios' => $this->propietarios()->count(),
            'inquilinos' => $this->inquilinos()->count(),
            'guardias' => $this->guardias()->count(),
            'areas_comunes' => $this->areasComunes()->count(),
            'camaras' => $this->camaras()->where('estado', 'activa')->count(),
            'pagos_pendientes' => $this->pagos()->where('estado', 'pendiente')->count(),
            'pagos_pagados' => $this->pagos()->where('estado', 'pagado')->count(),
        ];
    }

    /**
     * Obtiene la configuración con valores por defecto.
     */
    public function getConfiguracionCompletaAttribute(): array
    {
        $defaultConfig = [
            'tasa_mora' => 5.0,
            'dias_gracia' => 3,
            'notificaciones_automaticas' => true,
            'ciclo_facturacion_dia' => 1,
            'recordatorio_pago_dias' => 3,
            'bloqueo_reservas_mora' => true,
            'max_visitantes_por_unidad' => 5,
            'horario_areas_comunes_inicio' => '08:00',
            'horario_areas_comunes_fin' => '22:00',
            'reserva_max_dias_anticipacion' => 30,
            'cobro_tardio_automatico' => true,
            'multa_ruido' => 50.00,
            'multa_estacionamiento' => 100.00,
            'multa_mal_uso_areas' => 75.00,
            'backup_automatico' => true,
            'backup_frecuencia' => 'daily',
            'integracion_whatsapp' => false,
            'integracion_paypal' => true,
            'mostrar_camaras_publicas' => true,
            'requerir_aprobacion_registro' => true,
            'token_registro_expiracion_horas' => 48,
        ];

        $userConfig = $this->configuracion ?? [];

        return array_merge($defaultConfig, $userConfig);
    }

    /**
     * Obtiene un valor específico de configuración.
     */
    public function getConfig(string $key, $default = null)
    {
        $config = $this->configuracion_completa;
        return $config[$key] ?? $default;
    }

    /**
     * Actualiza la configuración del condominio.
     */
    public function updateConfig(array $config): bool
    {
        $currentConfig = $this->configuracion ?? [];
        $mergedConfig = array_merge($currentConfig, $config);

        return $this->update(['configuracion' => $mergedConfig]);
    }

    /**
     * Genera un token de registro para nuevos usuarios del condominio.
     */
    public function generateRegistrationToken(string $role = 'propietario', int $hours = 48): string
    {
        $token = hash('sha256', $this->id . $role . now()->timestamp . bin2hex(random_bytes(16)));

        cache()->put("condominio_registration_token_{$token}", [
            'condominio_id' => $this->id,
            'role' => $role,
            'expires_at' => now()->addHours($hours),
            'created_by' => auth()->id()
        ], now()->addHours($hours));

        return $token;
    }

    /**
     * Obtiene la URL de registro público para el condominio.
     */
    public function getRegistrationUrl(string $role = 'propietario'): string
    {
        $token = $this->generateRegistrationToken($role);
        return route('register.condominio', ['token' => $token]);
    }

    /**
     * Obtiene el enlace corto para WhatsApp.
     */
    public function getWhatsAppRegistrationUrl(string $role = 'propietario'): string
    {
        $token = $this->generateRegistrationToken($role, 72); // 3 días para WhatsApp
        $shortUrl = url()->route('register.condominio', ['token' => $token]);
        
        // En producción, aquí se integraría con un acortador de URLs
        return $shortUrl;
    }

    /**
     * Obtiene el número total de unidades por tipo.
     */
    public function getUnidadesPorTipoAttribute(): array
    {
        return [
            'departamento' => $this->unidades()->where('tipo', 'departamento')->count(),
            'cochera' => $this->unidades()->where('tipo', 'cochera')->count(),
            'deposito' => $this->unidades()->where('tipo', 'deposito')->count(),
            'combinado' => $this->unidades()->where('tipo', 'combinado')->count(),
        ];
    }

    /**
     * Obtiene los morosos del condominio.
     */
    public function getMorosos()
    {
        return $this->pagos()
            ->where('estado', 'pendiente')
            ->where('fecha_vencimiento', '<', now())
            ->with('user', 'unidad')
            ->get()
            ->groupBy('user_id');
    }

    /**
     * Calcula la morosidad total.
     */
    public function getMorosidadTotalAttribute(): float
    {
        return $this->pagos()
            ->where('estado', 'pendiente')
            ->where('fecha_vencimiento', '<', now())
            ->sum('monto');
    }

    /**
     * Obtiene los ingresos del mes actual.
     */
    public function getIngresosMesActualAttribute(): float
    {
        $startOfMonth = now()->startOfMonth();
        $endOfMonth = now()->endOfMonth();

        return $this->pagos()
            ->where('estado', 'pagado')
            ->whereBetween('fecha_pago', [$startOfMonth, $endOfMonth])
            ->sum('monto');
    }

    /**
     * Activa el condominio.
     */
    public function activate(): bool
    {
        return $this->update(['estado' => 'activo']);
    }

    /**
     * Suspende el condominio.
     */
    public function suspend(): bool
    {
        return $this->update(['estado' => 'suspendido']);
    }

    /**
     * Desactiva el condominio.
     */
    public function deactivate(): bool
    {
        return $this->update(['estado' => 'inactivo']);
    }

    /**
     * Scope para condominios activos.
     */
    public function scopeActive($query)
    {
        return $query->where('estado', 'activo');
    }

    /**
     * Scope para condominios por superadministrador.
     */
    public function scopeBySuperadmin($query, $superadminId)
    {
        return $query->where('superadministrador_id', $superadminId);
    }

    /**
     * Scope para buscar condominios.
     */
    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('nombre', 'like', "%{$search}%")
              ->orWhere('direccion', 'like', "%{$search}%")
              ->orWhere('email', 'like', "%{$search}%");
        });
    }

    /**
     * Crea una copia de seguridad del condominio.
     */
    public function createBackup(): array
    {
        $backupData = [
            'condominio' => $this->toArray(),
            'unidades' => $this->unidades()->get()->toArray(),
            'usuarios' => $this->usuarios()->get()->toArray(),
            'areas_comunes' => $this->areasComunes()->get()->toArray(),
            'camaras' => $this->camaras()->get()->toArray(),
            'pagos' => $this->pagos()->get()->toArray(),
            'backup_date' => now()->toDateTimeString(),
            'backup_version' => '1.0',
        ];

        $filename = "backup_condominio_{$this->id}_" . now()->format('Y-m-d_H-i-s') . '.json';
        $path = "backups/condominios/{$filename}";

        Storage::disk('local')->put($path, json_encode($backupData, JSON_PRETTY_PRINT));

        return [
            'success' => true,
            'filename' => $filename,
            'path' => $path,
            'size' => Storage::disk('local')->size($path),
            'download_url' => route('admin.superadmin.backup.download', ['filename' => $filename]),
        ];
    }

    /**
     * Boot del modelo.
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($condominio) {
            if (empty($condominio->configuracion)) {
                $condominio->configuracion = [
                    'tasa_mora' => 5.0,
                    'dias_gracia' => 3,
                ];
            }
        });

        static::created(function ($condominio) {
            // Crear áreas comunes por defecto
            $areasComunes = [
                ['nombre' => 'Salón Social', 'tipo' => 'salon', 'capacidad' => 50],
                ['nombre' => 'Piscina', 'tipo' => 'piscina', 'capacidad' => 20],
                ['nombre' => 'Gimnasio', 'tipo' => 'gimnasio', 'capacidad' => 10],
                ['nombre' => 'Zona de Parrillas', 'tipo' => 'parrillas', 'capacidad' => 15],
            ];

            foreach ($areasComunes as $area) {
                $condominio->areasComunes()->create($area);
            }

            // Notificar al superadministrador
            activity()
                ->causedBy($condominio->superadministrador)
                ->performedOn($condominio)
                ->log("Condominio {$condominio->nombre} creado exitosamente");
        });

        static::updated(function ($condominio) {
            activity()
                ->causedBy(auth()->user())
                ->performedOn($condominio)
                ->withProperties($condominio->getChanges())
                ->log("Condominio {$condominio->nombre} actualizado");
        });
    }
}